<?php
/**
 * RP_Tips_Functions
 *
 * @package RP_Tips_Functions
 * @since 1.0
 */

defined( 'ABSPATH' ) || exit;

class RP_Tips_Functions {
  public function __construct() {

    add_filter( 'rpress_purchase_form_before_submit' , array( $this, 'rpress_tips_html_checkout' ) );

    add_action( 'wp_ajax_rpress_add_tips', array( $this, 'rpress_add_tips') );
    add_action( 'wp_ajax_nopriv_rpress_add_tips', array( $this, 'rpress_add_tips') );

    add_action( 'init', array( $this, 'rpress_reset_tips') );

    add_action( 'rpress_add_email_tags', array( $this, 'add_tips_to_email_tag' ), 100 ) ;

    add_action( 'wp_enqueue_scripts', array( $this, 'rp_tips_scripts' ) );

  }

  /**
   * Add necessary css and js file
   *
   * @since  1.0
   */
  public function rp_tips_scripts() {

    // Enqueue style and script
    wp_enqueue_style( 'rp-tips-style', RP_TIPS_PLUGIN_URL . 'includes/assets/css/rp-tips.css' );
    wp_enqueue_script( 'rp-tips-script', RP_TIPS_PLUGIN_URL . 'includes/assets/js/rp-tips.js', ['jquery'], RP_TIPS_VERSION );

    // Tips type symbol
    $tip_type         = rpress_get_option( 'tips_type' );
    $tip_type_symbol  = ( $tip_type == 'percentage_value' ) ? '%' : '';

    // Localize the script with data
    $localize_data = array(
      'ajaxurl'          => admin_url( 'admin-ajax.php' ),
      'tip_type_symbol'  => $tip_type_symbol,
    );
    wp_localize_script( 'rp-tips-script', 'tips_script', $localize_data );
    
  }

  /**
   * Remove Fees on page refresh
   **/
  public function rpress_reset_tips () {
    
    if ( function_exists( 'rpress_get_cart_fees' ) ) {
      // Remove all fees from checkout
      RPRESS()->fees->remove_fee( 'tip' );
    }
  }

  /**
   * Add Tips option to checkout page
   **/
  public function rpress_tips_html_checkout() {

    $is_tip_enabled = rpress_get_option( 'enable_tips_on_checkout' );
    if ( $is_tip_enabled ) {

      // Get tips options
      $default_tips         = rpress_get_option( 'tips_values' );
      $default_tips_values  = explode( ',', $default_tips );
      $tips_text            = rpress_get_option( 'tips_text' );

      // Get the tips type
      $tip_type             = rpress_get_option( 'tips_type' );
      $tips_type_display    = rpress_get_option( 'tips_type_display' );
      $tip_type_symbol      = ( $tip_type == 'percentage_value' ) ? '%' : '';
      ?>
        <div class="rpress-tips">
          <div class="section-label top25">
            <a class="section-label-a">
              <span class="bold">
                <?php echo $tips_text ?> <?php if ( $tip_type == 'percentage_value' ) echo '(<span class="tip_percentage">0%</span>)' ?>
              </span>
              <b></b>
            </a>
            </div>
          <div class="uk-panel uk-panel-box">
            <ul class="tip-wrapper">
              <?php
                if ( !empty( $default_tips ) && $tips_type_display != 'manual_tips' ) {
                  // Loop through all tips value
                  foreach ( $default_tips_values as $default_tips_value ) {
                    ?>
                      <li>
                        <a class="tips" href="javascript:;" data-type="tip_type_<?php echo $tip_type ?>" data-tip="<?php echo $default_tips_value; ?>"><?php echo ( $tip_type == 'fixed_values' ) ? rpress_currency_filter( trim( $default_tips_value ) ) : trim( $default_tips_value ) . $tip_type_symbol; ?></a>
                      </li>
                    <?php
                  }
                }

                if ( $tips_type_display != 'tips_options' ) {
                  ?>
                    <li>
                      <input class="numeric_only rpress-input" type="number" value="" name="tip_value" min="1" id="manual_tip_value" placeholder="<?php _e( 'Custom Tip', 'restropress' ); ?>">
                    </li>
                  <?php
                }
              ?>
            </ul>
          </div>
        </div>
      <?php

    }

  }

  /**
   * Add ajax function to include tip to total price
   **/
  public function rpress_add_tips() {

    // Check if tip is set
    if ( empty( $_POST['tip'] ) || empty( $_POST['type'] ) ) return;

    // Remove all fees from checkout
    RPRESS()->fees->remove_fee( 'tip' );
    

    // Calculate the tip value
    $cart_sub_total   = rpress_get_cart_subtotal();

    // Get tax option
    $is_tax       = rpress_get_option( 'enable_taxes' );
    $is_tax_tips  = rpress_get_option( 'inculde_tax_on_tips' );


    // Check tips type
    if ( $_POST['type'] == 'tip_type_percentage_value' ) {
      $tip_percent  = $_POST['tip'];
      $tip_value    = ( $tip_percent / 100 ) * $cart_sub_total;
    }else {
      $tip_value    = $_POST['tip'];
      $tip_percent  = ( $tip_value / $cart_sub_total ) * 100;
    }

    if ( $is_tax && $is_tax_tips ) $tip_value = $tip_value + rpress_calculate_tax( $tip_value );

    // Add a tip to checkout
    if ( $tip_value > 0 ) 
      RPRESS()->fees->add_fee( $tip_value, 'Tips', 'tip', 'tip', $is_tax );

    $total  = rpress_get_cart_total();
    $return = array(
      'response'    => 'success',
      'total_plan'  => $total,
      'total'       => html_entity_decode( rpress_currency_filter( rpress_format_amount( $total ) ), ENT_COMPAT, 'UTF-8' ),
      'percentage'  => rpress_format_amount( $tip_percent ),
      'html'        => RP_Tips_Settings::rpress_get_tips_html( 'tip' ),
    );

    echo json_encode( $return );

    wp_die();

  }


  /**
   * Add email tag for tips
   **/
  public function add_tips_to_email_tag( $email_tags ) {
    $email_tag = 'tips';
    $tag_description = esc_html( 'Tips for the order', 'rp-tips' );

    $is_tip_enabled = rpress_get_option( 'enable_tips_on_checkout' );
    
    if ( $is_tip_enabled ) {

      rpress_add_email_tag( $email_tag, $tag_description, array( $this,  'rpress_email_tag_tips') );
    }

  }

  /**
   * The {tips} email tag
   */
  public function rpress_email_tag_tips( $payment_id ) {

    $tips_price = 0;

    if ( !empty( $payment_id ) ) {
      $tips_price = $this->tips_price( $payment_id );
    }

    return rpress_currency_filter( rpress_format_amount( $tips_price ) );
    
  }

  
  public function tips_price( $payment_id ) {
    $fees       = rpress_get_payment_fees( $payment_id, 'fee' );
    $tip_price  = '';
    
    foreach ( $fees as $fee ) {
      if ( $fee['id'] == 'tip' ) {
        $tip_price = $fee['amount'];
      }
    }

    return $tip_price;
  }

}

new RP_Tips_Functions();