<?php
/**
 * RP_Tips
 *
 * @package RP_Tips
 * @since 1.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main RP_Tips Class.
 *
 * @class RP_Tips
 */
class RP_Tips {

  /**
   * RestroPress version.
   *
   * @var string
   */
  public $version = '1.0';


  /**
   * The single instance of the class.
   *
   * @var RP_Tips
   * @since 1.0
   */
  protected static $_instance = null;

  /**
   * Main RP_Tips Instance.
   *
   * Ensures only one instance of RP_Tips is loaded or can be loaded.
   *
   * @since 1.0
   * @static
   * @return RP_Tips - Main instance.
   */
  public static function instance() {
    if ( is_null( self::$_instance ) ) {
      self::$_instance = new self();
    }
    return self::$_instance;
  }

  /**
   * RP_Tips Constructor.
   */
  public function __construct() {
    $this->define_constants();
    $this->includes();
    $this->init_hooks();
  }

  /**
   * Define constant if not already set.
   *
   * @param string      $name  Constant name.
   * @param string|bool $value Constant value.
   */
  private function define( $name, $value ) {
    if ( ! defined( $name ) ) {
      define( $name, $value );
    }
  }

  /**
   * Define Constants
   */
  private function define_constants() {
    $this->define( 'RP_TIPS_VERSION', $this->version );
    $this->define( 'RP_TIPS_PLUGIN_DIR', plugin_dir_path( RP_TIPS_FILE ) );
    $this->define( 'RP_TIPS_PLUGIN_URL', plugin_dir_url( RP_TIPS_FILE ) );
    $this->define( 'RP_TIPS_BASE', plugin_basename( RP_TIPS_FILE ) );
  }


  /**
   * Hook into actions and filters.
   *
   * @since 1.0
   */
  private function init_hooks() {
    add_action( 'admin_notices', array( $this, 'rp_tips_required_plugins' ) );

    add_filter( 'plugin_action_links_'.RP_TIPS_BASE, array( $this, 'rp_tips_settings_link' ) );

    add_action( 'plugins_loaded', array( $this, 'rp_tips_load_textdomain' ) );
  }


  /**
   * Load text domain
   *
   * @since 1.0
   */
  public function rp_tips_load_textdomain() {
    load_plugin_textdomain( 'rp-tips', false, dirname( plugin_basename( RP_TIPS_FILE ) ). '/languages/' );
  }


  /**
   * Include required files for settings
   *
   * @since 1.0
   */
  private function includes() {
    require_once RP_TIPS_PLUGIN_DIR . 'includes/admin/rp-tips-settings.php';
    require_once RP_TIPS_PLUGIN_DIR . 'includes/rp-tips-functions.php';
  }


  /**
   * Check plugin dependency
   *
   * @since 1.0
   */
  public function rp_tips_required_plugins() {
    if ( ! is_plugin_active( 'restropress/restro-press.php' ) ) {
      $plugin_link = 'https://wordpress.org/plugins/restropress/';
      
      echo '<div id="notice" class="error"><p>' . sprintf( __( 'RestroPress Tips Add-on requires <a href="%1$s" target="_blank"> RestroPress </a> plugin to be installed. Please install and activate it', 'rp-tips' ), esc_url( $plugin_link ) ).  '</p></div>';
      
      deactivate_plugins( '/rpress-tips/rpress-tips.php' );
    }
  }


  /**
   * Add settings link for the plugin
   *
   * @since 1.0
   */
  public function rp_tips_settings_link( $links ) {
    $link = admin_url( 'edit.php?post_type=fooditem&page=rpress-settings&tab=general&section=checkout_options' );
    $settings_link = sprintf( __( '<a href="%1$s">Settings</a>', 'rp-tips' ), esc_url( $link ) ); 
    array_unshift( $links, $settings_link ); 
    return $links;
  }

}